# discover_real_rho.py
import glob, os, numpy as np

def score_rho(arr):
    arr = np.asarray(arr, dtype=float).ravel()
    if arr.size < 8 or not np.all(np.isfinite(arr)): return None
    dif = np.diff(arr)
    mono = (dif >= 0).sum() / max(1, dif.size)
    span = float(arr.max() - arr.min()) if arr.size else 0.0
    return dict(n=int(arr.size), mono=float(mono), span=span)

def try_npy(p):
    try:
        a = np.load(p, allow_pickle=False)
        sc = score_rho(a)
        return [("ARRAY", None, sc)] if sc else []
    except Exception:
        return []

def try_npz(p):
    out = []
    try:
        z = np.load(p)
        for k in z.files:
            sc = score_rho(z[k])
            if sc: out.append(("NPZ", k, sc))
    except Exception:
        pass
    return out

def try_csv(p):
    out = []
    try:
        import pandas as pd
        df = pd.read_csv(p)
        for col in df.columns:
            lname = str(col).lower()
            if "rho" in lname or lname in ("rhos","rho_i","rho_base"):
                sc = score_rho(df[col].to_numpy())
                if sc: out.append(("CSV", col, sc))
        # If D & rho present, prefer 'rho'
        if "D" in df.columns and "rho" in df.columns:
            sc = score_rho(df["rho"].to_numpy()); 
            if sc: out.append(("CSV", "rho", sc))
    except Exception:
        pass
    return out

cands = []
patterns = ["data/**/*.*", "results/**/*.*", "artifacts/**/*.*", "**/*.*"]
seen = set()
for pat in patterns:
    for p in glob.glob(pat, recursive=True):
        if p in seen: continue
        seen.add(p)
        ext = os.path.splitext(p)[1].lower()
        rows = []
        if ext == ".npy": rows = try_npy(p)
        elif ext == ".npz": rows = try_npz(p)
        elif ext == ".csv": rows = try_csv(p)
        for typ, key, sc in rows:
            if sc and sc["n"] <= 100000:  # filter absurdly large
                cands.append((p, typ, key, sc))

# Rank by monotonicity, then length
cands.sort(key=lambda x: (x[3]["mono"], x[3]["n"]), reverse=True)
if not cands:
    print("NONE")
else:
    for p, typ, key, sc in cands[:20]:
        print(f"{p} [{typ}{':'+key if key else ''}]  n={sc['n']}  mono={sc['mono']:.3f}  span={sc['span']:.6g}")
